<?php

namespace App\Repositories;

use App\Models\User;
use App\Models\Factor;
use App\Enums\UserTypeEnum;
use App\Enums\PlanTypeEnum;
use App\Models\Specification;

class UserRepository
{
    const DEFAULT_PAGINATION = 15;

    public function getFactoryOwners()
    {
        return User::query()
            ->with('specification')
            ->factoryOwner()
            ->paginate(self::DEFAULT_PAGINATION);
    }

    public function getInvestors()
    {
        return User::query()
            ->with('specification')
            ->investor()
            ->paginate(self::DEFAULT_PAGINATION);
    }

    public function getStatistics(): array
    {
        $statistics = [];
        $statistics['users-count'] = User::query()->count();
        $statistics['factory-owners-count'] = User::query()->factoryOwner()->count();
        $statistics['investors-count'] = User::query()->investor()->count();
        return $statistics;
    }


    public function storeUser(array $userInformation, PlanTypeEnum $planType): User
    {
        $company = $userInformation['company'];

        $userPhone = $userInformation['phone'];
        $userType = UserTypeEnum::tryFromString($userInformation['type']);

        $user = $this->getUserByPhone($userPhone, $userType);

        $isUserRegisteredBefore = !empty($user);

        if ($isUserRegisteredBefore) {
            $user->factor()->delete();
            $user->specification()->delete();
        } else {
            $user = new User();
        }

        $user->{User::COLUMN_FULL_NAME} = $userInformation['full_name'];
        $user->{User::COLUMN_PHONE} = $userPhone;
        $user->{User::COLUMN_EMAIL} = $userInformation['email'];
        $user->{User::COLUMN_COMPANY_NAME} = $company['name'];
        $user->{User::COLUMN_COMPANY_ADDRESS} = $company['address'];
        $user->{User::COLUMN_TYPE} = $userType;
        $user->{User::COLUMN_PLAN} = $planType->value;
        $user->save();

        return $user;
    }

    public function storeSpecifications(int $userId, array $specifications): void
    {
        $specification = new Specification();

        $specification->{Specification::COLUMN_USER_ID} = $userId;
        $specification->{Specification::COLUMN_INVESTMENT} = $specifications['investment'];
        $specification->{Specification::COLUMN_PRODUCTION} = $specifications['production'];
        $specification->{Specification::COLUMN_DEMAND} = $specifications['demanded'];
        $specification->{Specification::COLUMN_WORK_SHIFTS} = $specifications['work_shifts'];
        $specification->{Specification::COLUMN_CONSUMPTION} = $specifications['consumption'];
        $specification->{Specification::COLUMN_REGION} = 1; // Isfahan Currently
        $specification->{Specification::COLUMN_RENT} = $specifications['rent'];

        $specification->save();
    }

    public function storeFactors(array $factors): void
    {
        Factor::query()->insert($factors);
    }

    private function getUserByPhone(string $userPhone, UserTypeEnum $userType): ?User
    {
        return User::query()
            ->where(User::COLUMN_TYPE, $userType)
            ->where(User::COLUMN_PHONE, $userPhone)
            ->first();
    }
}


